﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using RevisionAnalyser.Global;

namespace RevisionAnalyser.Controls
{
    public partial class RevisionSet : Component
    {
        private SortedDictionary<long, Revision> _revisions = new SortedDictionary<long, Revision>();

        [Description("Gets the first revision number that is stored."),
        Category("Values"),
        Browsable(true),
        ReadOnly(true)]
        public long FirstRevision
        {
            get
            {
                return RevisionCount == 0 ? 0 : _revisions.Keys.First<long>();
            }
        }

        [Description("Gets the last revision number that is stored."),
        Category("Values"),
        Browsable(true),
        ReadOnly(true)]
        public long LastRevision
        {
            get
            {
                return RevisionCount == 0 ? 0 : _revisions.Keys.Last<long>();
            }
        }

        [Description("Gets the number of revisions."),
        Category("Values"),
        Browsable(true),
        ReadOnly(true)]
        public int RevisionCount
        {
            get
            {
                return _revisions.Count;
            }
        }

        public RevisionSet()
        {
            InitializeComponent();
        }

        /// <summary>
        /// Adds a revision.
        /// </summary>
        /// <param name="rev">Revision number.</param>
        public void AddRevision(Revision rev, bool update)
        {
            if (!_revisions.ContainsKey(rev.ID))
            {
                _revisions.Add(rev.ID, rev);
                if (update)
                {
                    Updated(null, null);
                }
            }
            else
            {
                throw new Exception(String.Format("The revision number {0} was already added.", rev.ID));
            }
        }

        public void AddRevision(Revision rev)
        {
            AddRevision(rev, true);
        }

        /// <summary>
        /// Returns the date and time of a given revision number.
        /// </summary>
        /// <param name="id">Revision number.</param>
        /// <returns>Date and time of the revision.</returns>
        public Revision GetRevision(long id)
        {
            if (_revisions.ContainsKey(id))
            {
                return _revisions[id];
            }
            else
            {
                throw new NullReferenceException(String.Format("The revision number {0} does not exist.", id));
            }
        }

        public SortedDictionary<long, Revision> GetRevisionDictionary()
        {
            return _revisions;
        }

        public List<Revision> GetRevisionList()
        {
            return new List<Revision>(_revisions.Values);
        }

        /// <summary>
        /// Checks whether a revision number exists.
        /// </summary>
        /// <param name="id">Revision number to check for.</param>
        /// <returns>Boolean indicating whether the revision exists.</returns>
        public bool ContainsRevision(long id)
        {
            return _revisions.ContainsKey(id);
        }

        /// <summary>
        /// Removes a revision.
        /// </summary>
        /// <param name="id">Revision to remove.</param>
        /// <returns>Boolean indicating the success of the removal.</returns>
        public bool DeleteRevision(long id, bool update)
        {
            bool b = _revisions.Remove(id);
            if (update)
            {
                Updated(null, null);
            }
            return b;
        }

        public bool DeleteRevision(long id)
        {
            return DeleteRevision(id, true);
        }

        public void ClearRevisions()
        {
            _revisions.Clear();
            Updated(null, null);
        }

        public void Update()
        {
            Updated(null, null);
        }

        public event EventHandler Updated;
    }
}
